import React, { useState, useEffect } from "react";
import ReactPaginate from "react-paginate";
import Footer from "../../../components/Footer";
import Header from "../../../components/Header";
import AgentImage from "../../../assets/roberto-nickson-bwIqQ5qQhXs-unsplash.webp";
import LenderCard from "../../../components/LenderResultCard";
import {
  getLenderByCity,
  getLenderByCityAndName,
  getLenderByName,
  getLenderNames,
  getPremiumLenders,
  getLenderFullNames,
  getLenderByCityAndFullName,
} from "../../../utils/api";
import PlacesAutocomplete from "../../../components/PlacesAutocomplete";
import { useSearchParams } from "react-router-dom";

const Search = () => {
  const [agents, setAgents] = useState([]);
  const [isFeatured, setIsFeatured] = useState(false);
  const [isLoading, setLoading] = useState(false);

  const [searchParams, setSearchParams] = useSearchParams();

  // Search Inputs
  const [selectedName, setSelectedName] = useState("");
  const [formattedAddress, setFormattedAddress] = useState("");
  const [showNameDropdown, setShowNameDropdown] = useState(false);
  const [names, setNames] = useState([]);
  const [filteredNames, setFilteredNames] = useState([]);
  const [isFullName, setIsFullName] = useState(false);

  // Pagination
  const itemsPerPage = 9;
  const [currentItems, setCurrentItems] = useState([]);
  const [pageCount, setPageCount] = useState(0);
  const [itemOffset, setItemOffset] = useState(0);

  const premiumAgentsFirst = (data) => {
    const premium = data.filter((agent) => agent.isPremium === true);
    const regular = data.filter((agent) => !agent.isPremium);
    return premium.concat(regular);
  };

  useEffect(() => {
    const endOffset = itemOffset + itemsPerPage;
    setCurrentItems(agents.slice(itemOffset, endOffset));
    setPageCount(Math.ceil(agents.length / itemsPerPage));
  }, [itemOffset, agents]);

  const handlePageClick = (event) => {
    const newOffset = (event.selected * itemsPerPage) % agents.length;
    setItemOffset(newOffset);
  };

  const handleNameInputSelect = (name) => {
    setSelectedName(name);
    setNames([]);
    setFilteredNames([]);
    setShowNameDropdown(false);
    setIsFullName(true);
  };

  const filterName = (val) => {
    const filtered = names.filter((name) =>
      name.toLowerCase().startsWith(val.toLowerCase())
    );
    setFilteredNames(filtered);
  };

  const handleNameInputChange = (e) => {
    const name = e.target.value;
    setSelectedName(name);
    setIsFullName(false);

    if (name.trim() === "") {
      setShowNameDropdown(false);
      setNames([]);
    } else if (name.length === 1) {
      getLenderNames(name.trim())
        .then((res) => res.json())
        .then((data) => {
          setNames(data);
          setFilteredNames(data);
          setShowNameDropdown(true);
        });
    } else {
      filterName(name);
    }
  };

  const handleNameKeyDown = (e) => {
    const key = e.keyCode;
    const index = filteredNames.indexOf(selectedName);

    if (key === 40 && filteredNames[index + 1]) {
      setSelectedName(filteredNames[index + 1]);
    } else if (key === 38 && filteredNames[index - 1]) {
      setSelectedName(filteredNames[index - 1]);
    } else if (key === 13) {
      setShowNameDropdown(false);
      setNames([]);
      setFilteredNames([]);
      setIsFullName(true);
    }
  };

  const searchLenders = (city, name, isFull) => {
    setLoading(true);
    const searchBy = (city ? 1 : 0) + (name ? 2 : 0);

    const handleResult = (data) => {
      const list = data?.filterCities || data || [];
      setAgents(premiumAgentsFirst(list));
      setIsFeatured(false);
      setLoading(false);
      setItemOffset(0);
    };

    switch (searchBy) {
      case 1:
        getLenderByCity(city)
          .then((res) => res.json())
          .then(handleResult);
        break;
      case 2:
        (isFull ? getLenderFullNames(name) : getLenderByName(name))
          .then((res) => res.json())
          .then(handleResult);
        break;
      case 3:
        (isFull
          ? getLenderByCityAndFullName(city, name)
          : getLenderByCityAndName(city, name)
        )
          .then((res) => res.json())
          .then(handleResult);
        break;
      default:
        getPremiumLenders()
          .then((res) => res.json())
          .then((data) => {
            setAgents(data);
            setIsFeatured(true);
            setLoading(false);
          });
    }
  };

  const handleSearch = (e) => {
    e.preventDefault();
    const city = formattedAddress.trim();
    const name = selectedName.trim();

    const params = {};
    if (city) params.city = city;
    if (name) params.name = name;

    setSearchParams(params);
    searchLenders(city, name, isFullName);
  };

  // ✅ On mount: read URL & search
  useEffect(() => {
    const city = searchParams.get("city")?.trim() || "";
    const name = searchParams.get("name")?.trim() || "";

    setFormattedAddress(city);
    setSelectedName(name);
    const isFull = name.split(" ").length > 1;
    setIsFullName(isFull);

    searchLenders(city, name, isFull);
  }, []);

  if (isLoading) {
    return (
      <>
        <Header />
        <div className="min-height">Loading...</div>
        <Footer />
      </>
    );
  }

  return (
    <div>
      <Header />

      <div className="agent-search-header">
        <img
          className="agent-search-image"
          src={AgentImage}
          alt="Lender Banner"
        />
        <div className="agent-search-header-bg lender-search-header-bg">
          <h1>Search Find A Home Lenders</h1>

          <div className="agent-search-form lender-search-form">
            <div className="input-container">
              <PlacesAutocomplete
                setFormattedAddress={setFormattedAddress}
                formattedAddress={formattedAddress}
              />
            </div>
            <div className="input-container">
              <input
                className="agent-search-input"
                type="text"
                placeholder="Name"
                value={selectedName}
                onChange={handleNameInputChange}
                onKeyDown={handleNameKeyDown}
                autoComplete="off"
              />
              {showNameDropdown && (
                <div className="dropdown-items">
                  {filteredNames.map((name) => (
                    <div
                      key={name}
                      className={
                        name === selectedName
                          ? "dropdown-item highlight"
                          : "dropdown-item"
                      }
                      onClick={() => handleNameInputSelect(name)}
                    >
                      <p>{name}</p>
                    </div>
                  ))}
                </div>
              )}
            </div>
            <button
              className="agent-search-btn text-shadow"
              onClick={handleSearch}
            >
              Search Now
            </button>
          </div>
        </div>
      </div>

      <main className="agent-search-results">
        <h2>{isFeatured ? "Featured Lenders" : "Lenders"}</h2>
        <div className="agent-search-results-container">
          {currentItems.map((item) => (
            <LenderCard key={item.id} agent={item} />
          ))}
        </div>
        <div className="pagination-container">
          <ReactPaginate
            nextLabel="next >"
            onPageChange={handlePageClick}
            pageRangeDisplayed={3}
            marginPagesDisplayed={2}
            pageCount={pageCount}
            previousLabel="< previous"
            pageClassName="page-item"
            pageLinkClassName="page-link"
            previousClassName="page-item"
            previousLinkClassName="page-link"
            nextClassName="page-item"
            nextLinkClassName="page-link"
            breakLabel="..."
            breakClassName="page-item"
            breakLinkClassName="page-link"
            containerClassName="pagination"
            activeClassName="active"
            renderOnZeroPageCount={null}
          />
        </div>
      </main>

      <Footer />
    </div>
  );
};

export default Search;
