const router = require("express").Router();
const { Agent, Uploads, User, Notices } = require("../../models");
const { Op } = require("sequelize");
const { cityStartsWith } = require("../utils/agentHelpers");

router.get("/", (req, res) => {
  Agent.findAll({
    where: {
      isDeleted: false,
    },
  })
    .then((allOficers) => {
      const filteredAgents = allOficers.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

router.get("/recently-added", (req, res) => {
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
    },
    limit: 50,
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Premium Agents
*/
router.get("/premium", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
      isPremium: true,
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by City
*/
// router.get("/city/:term", (req, res) => {
//   // Get all Agents including its associated User and Student data
//   Agent.findAll({
//     where: {
//       isDeleted: false,
//       [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
//       areas_served: {
//         [Op.substring]: req.params.term,
//       },
//     },
//     order: [["id", "DESC"]],
//   })
//     .then((allAgents) => {
//       const filtered = cityStartsWith(allAgents, req.params.term);

//       res.json(filtered);
//     })
//     .catch((err) => {
//       console.log(err);
//       res.status(500).json(err);
//     });
// });

const stateMap = {
  alabama: "al",
  alaska: "ak",
  arizona: "az",
  arkansas: "ar",
  california: "ca",
  colorado: "co",
  connecticut: "ct",
  delaware: "de",
  florida: "fl",
  georgia: "ga",
  hawaii: "hi",
  idaho: "id",
  illinois: "il",
  indiana: "in",
  iowa: "ia",
  kansas: "ks",
  kentucky: "ky",
  louisiana: "la",
  maine: "me",
  maryland: "md",
  massachusetts: "ma",
  michigan: "mi",
  minnesota: "mn",
  mississippi: "ms",
  missouri: "mo",
  montana: "mt",
  nebraska: "ne",
  nevada: "nv",
  "new hampshire": "nh",
  "new jersey": "nj",
  "new mexico": "nm",
  "new york": "ny",
  "north carolina": "nc",
  "north dakota": "nd",
  ohio: "oh",
  oklahoma: "ok",
  oregon: "or",
  pennsylvania: "pa",
  "rhode island": "ri",
  "south carolina": "sc",
  "south dakota": "sd",
  tennessee: "tn",
  texas: "tx",
  utah: "ut",
  vermont: "vt",
  virginia: "va",
  washington: "wa",
  "west virginia": "wv",
  wisconsin: "wi",
  wyoming: "wy",
};

const normalize = (text = "") => {
  return text
    .toLowerCase()
    .replace(/ee\.?\s?uu\.?/gi, "usa")
    .replace(/estados unidos/gi, "usa")
    .replace(/états-unis/gi, "usa")
    .replace(/\s+/g, " ")
    .trim();
};

const buildSearchVariants = (term) => {
  let normalized = normalize(term);

  let withAbbr = normalized;
  for (const [full, abbr] of Object.entries(stateMap)) {
    const regex = new RegExp(`\\b${full}\\b`, "gi");
    withAbbr = withAbbr.replace(regex, abbr);
  }
  let withFull = normalized;
  for (const [full, abbr] of Object.entries(stateMap)) {
    const regex = new RegExp(`\\b${abbr}\\b`, "gi");
    withFull = withFull.replace(regex, full);
  }

  return Array.from(new Set([normalized, withAbbr, withFull]));
};

router.get("/city/:term", async (req, res) => {
  try {
    const variants = buildSearchVariants(req.params.term);

    const allAgents = await Agent.findAll({
      where: { isDeleted: false },
      order: [["id", "DESC"]],
    });

    const matched = allAgents.filter((agent) => {
      const area = normalize(agent.areas_served);
      return variants.some((v) => area.includes(v));
    });

    res.json({ filterCities: matched });
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: "Internal server error" });
  }
});
/*
  Agent Search Page - Search by City (Search Bar)
*/
router.get("/city-searchbar/:term", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
      areas_served: {
        [Op.substring]: req.params.term,
      },
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      const filtered = cityStartsWith(allAgents, req.params.term);

      res.json(filtered.uniqueCities);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Name (Search Bar)
*/
router.get("/name-searchbar/:name", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
      [Op.or]: [
        {
          first_name: {
            [Op.startsWith]: req.params.name,
          },
        },
        {
          last_name: {
            [Op.startsWith]: req.params.name,
          },
        },
      ],
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      const names = [];

      for (let i = 0; i < filteredAgents.length; i++) {
        const el = filteredAgents[i];
        names.push(`${el.first_name} ${el.last_name}`);
      }

      res.json(names);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Name
*/
router.get("/email/:email", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
      email: {
        [Op.substring]: req.params.email,
      },
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Name
*/
router.get("/name/:name", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
      [Op.or]: [
        {
          first_name: {
            [Op.startsWith]: req.params.name,
          },
        },
        {
          last_name: {
            [Op.startsWith]: req.params.name,
          },
        },
      ],
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Full Name
*/
router.get("/full-name/:name", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      full_name: req.params.name,
      [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Name & City
*/
// router.get("/city/:term/name/:name", (req, res) => {
//   // Get all Agents including its associated User and Student data
//   Agent.findAll({
//     where: {
//       isDeleted: false,
//       [Op.or]: [{ primary_language: "English" }, { primary_language: "Both" }],
//       areas_served: {
//         [Op.substring]: req.params.term,
//       },
//       [Op.or]: [
//         {
//           first_name: {
//             [Op.startsWith]: req.params.name,
//           },
//         },
//         {
//           last_name: {
//             [Op.startsWith]: req.params.name,
//           },
//         },
//       ],
//     },
//     order: [["id", "DESC"]],
//   })
//     .then((allAgents) => {
//       const filtered = cityStartsWith(allAgents, req.params.term);

//       res.json(filtered);
//     })
//     .catch((err) => {
//       console.log(err);
//       res.status(500).json(err);
//     });
// });
router.get("/city/:term/name/:name", (req, res) => {
  const cityTerm = req.params.term;
  const nameSearch = req.params.name;
  const cityVariants = buildSearchVariants(cityTerm);
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [
        {
          first_name: {
            [Op.startsWith]: nameSearch,
          },
        },
        {
          last_name: {
            [Op.startsWith]: nameSearch,
          },
        },
      ],
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      const filtered = allAgents.filter((agent) => {
        const area = normalize(agent.areas_served);
        return cityVariants.some((v) => area.includes(v));
      });

      res.json({ filterCities: filtered });
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Full Name & City
*/
// router.get("/city/:term/full-name/:name", (req, res) => {
//   // Get all Agents including its associated User and Student data
//   Agent.findAll({
//     where: {
//       isDeleted: false,
//       areas_served: {
//         [Op.substring]: req.params.term,
//       },
//       full_name: req.params.name,
//     },
//     order: [["id", "DESC"]],
//   })
//     .then((allAgents) => {
//       const filtered = cityStartsWith(allAgents, req.params.term);

//       res.json(filtered);
//     })
//     .catch((err) => {
//       console.log(err);
//       res.status(500).json(err);
//     });
// });
router.get("/city/:term/full-name/:name", (req, res) => {
  const cityTerm = req.params.term;
  const fullName = req.params.name;

  const cityVariants = buildSearchVariants(cityTerm); // handles Spanish/English spellings etc.

  Agent.findAll({
    where: {
      isDeleted: false,
      full_name: fullName,
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Manual filter for areas_served to support multilingual spellings
      const filtered = allAgents.filter((agent) => {
        const served = normalize(agent.areas_served || "");
        return cityVariants.some((variant) =>
          served.includes(normalize(variant))
        );
      });

      // ✅ Respond same way as before — frontend does not need to change
      res.json({ filterCities: filtered });
    })
    .catch((err) => {
      console.error("Agent fetch error:", err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Premium Agents
*/
router.get("/busca/premium", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      isPremium: true,
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Recently Added Agents
*/
router.get("/busca/recently-added", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
    },
    limit: 50,
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by City
*/
router.get("/busca/city/:term", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      areas_served: {
        [Op.substring]: req.params.term,
      },
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      const filtered = cityStartsWith(allAgents, req.params.term);

      res.json(filtered);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by City (Search Bar)
*/
router.get("/busca/city-searchbar/:term", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      areas_served: {
        [Op.substring]: req.params.term,
      },
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      const filtered = cityStartsWith(allAgents, req.params.term);

      res.json(filtered.uniqueCities);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Name (Search Bar)
*/
router.get("/busca/name-searchbar/:name", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      [Op.or]: [
        {
          first_name: {
            [Op.startsWith]: req.params.name,
          },
        },
        {
          last_name: {
            [Op.startsWith]: req.params.name,
          },
        },
      ],
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      const names = [];

      for (let i = 0; i < filteredAgents.length; i++) {
        const el = filteredAgents[i];
        names.push(`${el.first_name} ${el.last_name}`);
      }

      res.json(names);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Name
*/
router.get("/busca/email/:email", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      email: {
        [Op.substring]: req.params.email,
      },
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Name
*/
router.get("/busca/name/:name", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      [Op.or]: [
        {
          first_name: {
            [Op.startsWith]: req.params.name,
          },
        },
        {
          last_name: {
            [Op.startsWith]: req.params.name,
          },
        },
      ],
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Full Name
*/
router.get("/busca/full-name/:name", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      full_name: req.params.name,
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      // Filter through Users, do not return "deleted" users
      const filteredAgents = allAgents.filter(
        (Agent) => Agent.isDeleted !== true
      );
      res.json(filteredAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Name & City
*/
router.get("/busca/city/:term/name/:name", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      areas_served: {
        [Op.substring]: req.params.term,
      },
      [Op.or]: [
        {
          first_name: {
            [Op.startsWith]: req.params.name,
          },
        },
        {
          last_name: {
            [Op.startsWith]: req.params.name,
          },
        },
      ],
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      const filtered = cityStartsWith(allAgents, req.params.term);

      res.json(filtered);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Search Page - Search by Full Name & City
*/
router.get("/busca/city/:term/full-name/:name", (req, res) => {
  // Get all Agents including its associated User and Student data
  Agent.findAll({
    where: {
      isDeleted: false,
      [Op.or]: [{ primary_language: "Spanish" }, { primary_language: "Both" }],
      areas_served: {
        [Op.substring]: req.params.term,
      },
      full_name: req.params.name,
    },
    order: [["id", "DESC"]],
  })
    .then((allAgents) => {
      const filtered = cityStartsWith(allAgents, req.params.term);

      res.json(filtered);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Profile Page
*/
router.get("/:agent_id", (req, res) => {
  Agent.findOne({
    where: {
      id: req.params.agent_id,
      isDeleted: false,
    },
    include: [{ model: Uploads }],
  })
    .then((allAgents) => {
      res.json(allAgents);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Create Agent
*/
router.post("/", (req, res) => {
  /*
    expects:
      {
        first_name: STRING,
        last_name: STRING,
        phone_number: STRING,
        email: STRING,
        languages_spoken: STRING,
        primary_language: STRING,
        areas_served: STRING,
        brokerage: STRING,
        state_license_numbers: STRING,
        profile_description: STRING,
        image: STRING,
      },
  */
  req.body.full_name = `${req.body.first_name} ${req.body.last_name}`;

  Agent.create(req.body)
    .then((newAgent) => {
      res.json({ newAgent });
    })
    .catch((err) => {
      console.log(err);

      res.status(500).json(err);
    });
});

/*
  Edit Agent Profile Page
*/
router.put("/:id", (req, res) => {
  // update a Agent by its `id` value
  // expects {columnName: value}
  Agent.update(req.body, {
    where: {
      id: req.params.id,
    },
  })
    .then((udpatedAgent) => {
      if (!udpatedAgent[0]) {
        res
          .status(404)
          .json({ message: "There is no Agent found with this id" });
        return;
      }
      res.json(udpatedAgent);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
   Super User Delete functionality
*/
// router.delete("/:id", (req, res) => {
//   // "Delete" a Agent by id - Update the boolean value to mark as "deleted"
//   // expects {id: INTEGER}
//   Agent.update(
//     { isDeleted: true },
//     {
//       where: {
//         id: req.params.id,
//       },
//     }
//   )
//     .then((deletedAgent) => {
//       if (!deletedAgent) {
//         res
//           .status(404)
//           .json({ message: "There are no Agents found with this id" });
//         return;
//       }
//       res.json(deletedAgent);
//     })
//     .catch((err) => {
//       console.log(err);
//       res.status(500).json(err);
//     });
// });
router.delete("/:id", async (req, res) => {
  try {
    const agent = await Agent.findOne({
      where: {
        id: req.params.id,
      },
    });

    if (!agent) {
      return res.status(404).json({ message: "No agent found with this ID." });
    }
    await Agent.destroy({
      where: {
        id: req.params.id,
      },
    });
    await Notices.destroy({
      where: {
        agent_id: req.params.id,
      },
    });
    await User.destroy({
      where: {
        email: agent.email,
      },
    });

    res.json({ message: "Agent and associated user deleted successfully." });
  } catch (err) {
    console.error("Delete error:", err);
    res.status(500).json({ message: "Server error while deleting agent." });
  }
});

module.exports = router;
