const router = require("express").Router();
const fetch = require("node-fetch");
const { filterPts } = require("../utils/filterPts.js");
const {
  filterCounties,
  filterZips,
  filterCities,
} = require("../utils/omnibarFilter.js");

// DB Model Routes and Prefixes
const agentRoutes = require("./agents-routes.js");
const userRoutes = require("./user-routes.js");
const uploadRoutes = require("./uploads-routes.js");
const noticeRoutes = require("./notices-routes.js");
const csvRoutes = require("./csv-routes.js");
const lenderRoutes = require("./lender-routes.js");
const emailRoutes = require("./email-routes.js");
router.use("/agents", agentRoutes);
router.use("/users", userRoutes);
router.use("/uploads", uploadRoutes);
router.use("/notices", noticeRoutes);
router.use("/csv", csvRoutes);
router.use("/lenders", lenderRoutes);
router.use("/email", emailRoutes);

// *** Comment this out when adding to cPanel ***
require("dotenv").config();
// *** REPLACE WITH REAL KEYS WHEN ADDING TO CPANEL ***
const GoogleMapsKey = "AIzaSyCWsRqMQ1TS9aXMvuDv7NlL3fDzjpeA4mE";
const IDXBrokerKey = "vGry9hs7gE5B0qcogddHWc";
const FollowUpBossKey =
  "Basic ZmthXzBJYWlNbkFaU05kSmlTWUU4RHFPMnRCNUF4ZUU0QnRIbEY6";
const BuscaFollowUpBossKey =
  "Basic ZmthXzBoV25UOEx6Z21SUHoxWEhSbVhpQ2pDM0l1SGFQWGlHUEY6";

// File's Prefixed Path: "/api"

// --------- GEOCODING, LAT/LNG ------------
// GET lat/lng data (Google Maps) for Results Page Map
router.post("/coords", async (req, res) => {
  const location = req.body.place;

  const reverseGeocodeUrl = `https://maps.googleapis.com/maps/api/geocode/json?address=${location}&key=${GoogleMapsKey}`;

  const response = await fetch(reverseGeocodeUrl, {
    method: "get",
    headers: {
      "Content-Type": "application/json",
    },
  });

  if (!response.ok)
    throw new Error(
      `unexpected response ${response.status}: ${response.statusText}`
    );

  const data = await response.json();

  if (!data) {
    res
      .status(404)
      .json({ message: "No data returned from Google's fetch request" });
  } else {
    res.status(200).json(data);
  }
});

// --------- GEOCODING, PLACE ------------
// GET City Name (Google Maps) for Homepage
router.post("/userLocation", async (req, res) => {
  const latitude = req.body.latitude;
  const longitude = req.body.longitude;

  const reverseGeocodeUrl = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${latitude},${longitude}&result_type=locality&key=${GoogleMapsKey}`;

  const response = await fetch(reverseGeocodeUrl, {
    method: "get",
    headers: {
      "Content-Type": "application/json",
    },
  });

  if (!response.ok)
    throw new Error(
      `unexpected response ${response.status}: ${response.statusText}`
    );

  const data = await response.json();

  if (!data) {
    res
      .status(404)
      .json({ message: "No data returned from Google's fetch request" });
  } else {
    res.status(200).json(data);
  }
});

// --------- GET MLS ID'S ------------
// GET MLS list for the Homepage + Footer
router.get("/mls", async (req, res) => {
  /* Returns and array of approved MLS objs for our account: [{ id: "b022", name: "realtracs MlS" },] */
  const url = `https://api.idxbroker.com/mls/approvedmls`;

  const response = await fetch(url, {
    method: "get",
    headers: {
      "Content-Type": "application/x-www-form-urlencoded",
      accesskey: IDXBrokerKey,
    },
  });

  if (!response.ok)
    throw new Error(
      `unexpected response ${response.status}: ${response.statusText}`
    );

  const data = await response.json();

  if (!data) {
    res.status(404).json({ message: "Unable to find MLS's for this account" });
    return;
  }

  res.status(200).json(data);
});

// --------- GET OMNI BAR ------------
// GET cities/zips/counties by MLS id for the Omni Bar
router.get("/omnibar/:mlsId", async (req, res) => {
  const mlsId = req.params.mlsId;

  Promise.all([getCities(mlsId), getZips(mlsId), getCounties(mlsId)])
    .then((data) => {
      const combine = data.flat(); // Combine the returned arrays
      res.status(200).json(combine);
    })
    .catch((err) => {
      console.log(err);
      res
        .status(400)
        .json({ message: "Something has gone wrong with this request" });
    });
});

// --------- GET PROPERTY FILTERS ------------
// GET Property Types by MLS id for the Filter Bar
router.get("/propTypes/:mlsId", async (req, res) => {
  const mlsId = req.params.mlsId;

  const url = `https://api.idxbroker.com/mls/propertytypes/${mlsId}`;

  const response = await fetch(url, {
    method: "get",
    headers: {
      "Content-Type": "application/x-www-form-urlencoded",
      accesskey: IDXBrokerKey,
    },
  });

  if (!response.ok)
    throw new Error(
      `unexpected response ${response.status}: ${response.statusText}`
    );

  const data = await response.json();

  if (!data) {
    res
      .status(404)
      .json({ message: "Unable to find Property types for this MLS" });
    return;
  }

  const propertytypes = filterPts(mlsId, data); // Filter Property Types

  res.status(200).json(propertytypes);
});

// --------- OMNIBAR CITIES ------------
const getCities = async (mlsId) => {
  const url = `https://api.idxbroker.com/mls/cities/${mlsId}`;

  const response = await fetch(url, {
    method: "get",
    headers: {
      "Content-Type": "application/x-www-form-urlencoded",
      accesskey: IDXBrokerKey,
    },
  });

  if (!response.ok)
    throw new Error(
      `unexpected response ${response.status}: ${response.statusText}`
    );

  const data = await response.json();

  if (!data) {
    res.status(404).json({ message: "Unable to find cities for this MLS" });
    return;
  }

  const cities = filterCities(mlsId, data); // Filter

  return cities;
};

// --------- OMNIBAR ZIP CODES ------------
const getZips = async (mlsId) => {
  const url = `https://api.idxbroker.com/mls/zipcodes/${mlsId}`;

  const response = await fetch(url, {
    method: "get",
    headers: {
      "Content-Type": "application/x-www-form-urlencoded",
      accesskey: IDXBrokerKey,
    },
  });

  if (!response.ok)
    throw new Error(
      `unexpected response ${response.status}: ${response.statusText}`
    );

  const data = await response.json();

  if (!data) {
    res.status(404).json({ message: "Unable to find cities for this MLS" });
    return;
  }

  const zips = filterZips(mlsId, data); // Filter

  return zips;
};

// --------- OMNIBAR COUNTIES ------------
const getCounties = async (mlsId) => {
  const url = `https://api.idxbroker.com/mls/counties/${mlsId}`;

  const response = await fetch(url, {
    method: "get",
    headers: {
      "Content-Type": "application/x-www-form-urlencoded",
      accesskey: IDXBrokerKey,
    },
  });

  if (!response.ok)
    throw new Error(
      `unexpected response ${response.status}: ${response.statusText}`
    );

  const data = await response.json();

  if (!data) {
    res.status(404).json({ message: "Unable to find cities for this MLS" });
    return;
  }

  const counties = filterCounties(mlsId, data); // Filter

  return counties;
};

// --------- RUUSTER POST API FOR NEW LEAD (CONTACT FORM) ------------
router.post("/sendNewLead", async (req, res) => {
  // Steps to set up integration: https://docs.followupboss.com/docs/start-here-brand-new-integration
  // Endpoint Docs: https://docs.followupboss.com/reference/events-post
  const firstName = req.body.firstName;
  const lastName = req.body.lastName;
  const message = req.body.message;
  const email = req.body.email;
  const phoneNumber = req.body.phoneNumber;

  const response = await fetch("https://api.followupboss.com/v1/events", {
    method: "POST",
    headers: {
      accept: "application/json",
      "content-type": "application/json",
      authorization: FollowUpBossKey,
    },
    body: JSON.stringify({
      person: {
        contacted: false,
        firstName: firstName,
        lastName: lastName,
        emails: [{ value: email }],
        phones: [{ value: phoneNumber }],
      },
      source: "findahomeinusa.com",
      system: "FindAHomeInUsa",
      type: "General Inquiry",
      message: message,
    }),
  });

  if (!response.ok) {
    res.status(404).json({
      message: `unexpected response ${response.status}: ${response.statusText}`,
    });
    return;
  }

  const data = await response.json();

  if (!data) {
    res.status(404).json({ message: "Unable to find MLS's for this account" });
    return;
  }

  res.status(200).json(data);
});

// --------- BUSCA TU CASA RUUSTER POST API FOR NEW LEAD (CONTACT FORM) ------------
router.post("/busca/sendNewLead", async (req, res) => {
  // Steps to set up integration: https://docs.followupboss.com/docs/start-here-brand-new-integration
  // Endpoint Docs: https://docs.followupboss.com/reference/events-post
  const firstName = req.body.firstName;
  const lastName = req.body.lastName;
  const message = req.body.message;
  const email = req.body.email;
  const phoneNumber = req.body.phoneNumber;

  const response = await fetch("https://api.followupboss.com/v1/events", {
    method: "POST",
    headers: {
      accept: "application/json",
      "content-type": "application/json",
      authorization: BuscaFollowUpBossKey,
    },
    body: JSON.stringify({
      person: {
        contacted: false,
        firstName: firstName,
        lastName: lastName,
        emails: [{ value: email }],
        phones: [{ value: phoneNumber }],
      },
      source: "buscatucasa.com",
      system: "BuscaTuCasa",
      type: "General Inquiry",
      message: message,
    }),
  });

  if (!response.ok) {
    res.status(404).json({
      message: `unexpected response ${response.status}: ${response.statusText}`,
    });
    return;
  }

  const data = await response.json();

  if (!data) {
    res.status(404).json({ message: "Unable to find MLS's for this account" });
    return;
  }

  res.status(200).json(data);
});

router.use((req, res) => {
  res.send("<h1>Wrong Route!</h1>");
});

module.exports = router;
