const router = require("express").Router();
const { User, Agent, PasswordReset, Lender } = require("../../models");
const bcrypt = require("bcryptjs");
const nodemailer = require("nodemailer");
const AppError = require("../../utils/appError");
require("dotenv").config();
// import sign token function from auth
const { generateToken } = require("../../utils/auth");

// The `/api/users` endpoint

/*
  Currently Unused
*/
router.get("/", (req, res) => {
  User.findAll({
    include: [{ model: Agent }],
    attributes: { exclude: ["password"] },
    required: false,
  })
    .then((allUsers) => {
      res.json({ data: allUsers });
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Agent Portal
*/
router.get("/:id", (req, res) => {
  User.findOne({
    where: {
      id: req.params.id,
      isDeleted: false,
    },
    include: [
      { model: Agent, required: false },
      { model: Lender, required: false },
    ],
    attributes: { exclude: ["password"] },
  })
    .then((singleUser) => {
      if (!singleUser) {
        res
          .status(404)
          .json({ message: "There is no User found with this id" });
        return;
      }

      res.json(singleUser);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  Create User
*/
router.post("/", (req, res) => {
  /*
    expects:
      {
        name: STRING,
        email: STRING,
        password: STRING,
        user_type: STRING,
      },
  */
  User.create(req.body)
    .then((newUser) => {
      res.json(newUser);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
  User Login
*/
router.post("/login", (req, res) => {
  /*
    expects:
      {
        email: STRING,
        password: STRING,
      },
  */
  User.findOne({
    where: {
      email: req.body.email,
    },
  })
    .then((loggedInUser) => {
      // Handle if user doesn't exist or if user has been "deleted"
      if (!loggedInUser) {
        res
          .status(404)
          .json({ message: "There is no account found with this email." });
        return;
      }

      // Check if the Password is corerct
      const validPassword = loggedInUser.checkPassword(req.body.password);
      if (!validPassword) {
        res
          .status(400)
          .json({ message: "Incorrect password for this account." });
        return;
      }

      const user_type = loggedInUser.dataValues.user_type;

      // Send the user_type (used to navigate on login) and token to the client
      const token = generateToken(loggedInUser);
      res.json({ token, user_type });
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
   User Portal Page
*/
router.put("/:id", async (req, res) => {
  // IF the user opts to update their password, hash the password before saving
  if (req.body.password) {
    hashedPassword = await bcrypt.hash(req.body.password, 7);
    req.body.password = hashedPassword;
  }

  // update a User by its `id` value
  // expects {columnName: value}
  User.update(req.body, {
    where: {
      isDeleted: false,
      id: req.params.id,
    },
  })
    .then((udpatedUser) => {
      if (!udpatedUser[0]) {
        res
          .status(404)
          .json({ message: "There is no User found with this id" });
        return;
      }
      res.json(udpatedUser);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});

/*
   Currently Unused
*/
router.delete("/:id", (req, res) => {
  // "Delete" a User by id - Update the boolean value to mark as "deleted"
  // expects {id: INTEGER}
  User.update(
    { isDeleted: true },
    {
      where: {
        id: req.params.id,
      },
      attributes: { exclude: ["password"] },
    }
  )
    .then((deletedUser) => {
      if (!deletedUser) {
        res
          .status(404)
          .json({ message: "There is no User found with this id" });
        return;
      }
      res.json(deletedUser);
    })
    .catch((err) => {
      console.log(err);
      res.status(500).json(err);
    });
});
//Reset Password
router.post("/forgot-password", async (req, res, next) => {
  if (!req.body || !req.body.email) {
    return next(new AppError("Email address is required", 400));
  }

  try {
    // find the user by email
    const user = await User.findOne({ where: { email: req.body.email } });

    if (!user) {
      return res
        .status(404)
        .json({ message: "Email does not exist in our database." });
    }

    // generate OTP and set expiry time
    const otp = Math.floor(1000 + Math.random() * 9000);
    const expiration = new Date();
    expiration.setMinutes(expiration.getMinutes() + 20);

    // create a record in PasswordReset model
    console.log("PasswordReset model:", PasswordReset);
    await PasswordReset.create({
      user_id: user.id,
      email: user.email,
      otp: otp,
      expiration: expiration,
    });

    // set up nodemailer transporter
    const transporter = nodemailer.createTransport({
      service: "gmail",
      auth: {
        user: "help@brainjar.net",
        pass: "voeujiyymzlthwdc",
      },
    });

    // mail options

    const mailOptions = {
      from: "camila@buscatucasa.com",
      to: user.email,
      subject: "Password Reset OTP",
      //we can style this
      text: `Your OTP for password reset is: ${otp}. This OTP will expire in 20 minutes.`,
    };

    // send email
    transporter.sendMail(mailOptions, (error, info) => {
      if (error) {
        return next(new AppError("Error sending email: " + error.message, 500));
      } else {
        res.json({
          message: "Your OTP has been sent to the email address.",
        });
      }
    });
  } catch (err) {
    console.error("Error:", err);
    return next(new AppError("Server error: " + err.message, 500));
  }
});
//otp validation
router.post("/check-otp", async (req, res) => {
  /**
   * Check OTP Validity
   * Pages Used: Password Reset Page
   * expects {otp: STRING} in the request body
   */

  const { otp } = req.body;
  // check if the OTP is provided if not, return a 400 status with a message.

  if (!otp) {
    return res.status(400).json({ message: "OTP is required" });
  }

  try {
    // attempt to find the OTP in the database.
    const passwordReset = await PasswordReset.findOne({ where: { otp } });
    // if the OTP does not exist in the database, return a 404 status with a message.

    if (!passwordReset) {
      return res.status(404).json({ message: "Invalid OTP" });
    }
    // get the current date and time.
    const now = new Date();
    // check if the current date and time is later than the OTP's expiration date

    if (now > passwordReset.otpExpire) {
      return res.status(400).json({ message: "OTP has expired" });
    }
    // if the OTP is valid and not expired return a 200 status with a success message

    return res.status(200).json({ message: "OK" });
  } catch (err) {
    console.log(err);
    res.status(500).json({ message: "Server error: " + err.message });
  }
});
//reset password
router.post("/update-password", async (req, res, next) => {
  const { email, otp, newPassword } = req.body;

  if (!email || !otp || !newPassword) {
    return res.status(400).json({ message: "All fields are required" });
  }

  try {
    // find the OTP record and verify it's valid
    const passwordReset = await PasswordReset.findOne({
      where: { otp, email },
    });
    const now = new Date();

    if (!passwordReset || now > passwordReset.otpExpire) {
      return res.status(400).json({ message: "Invalid or expired OTP" });
    }

    // hash the new password before saving
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    // update the user's password
    await User.update({ password: hashedPassword }, { where: { email } });

    res.status(200).json({ message: "Password successfully updated" });
  } catch (err) {
    console.log(err);
    res.status(500).json({ message: "Server error: " + err.message });
  }
});
module.exports = router;
