/*
Data from the IDX Broker API returns all available filters. Currently, we are only looking for the following:
Homes (Residential), Single Family Residence, Townhomes, Condo, Manufactured on Land, Residential Lease, 
Commercial Lease, Commercial Sale, Farms, and Lots + Land.

This helper functions filter out the extra objects and returns an array of formatted objs like: 
{
  mlsId: mlsId, 
  values: [
            {
              "display": "Commercial Sale", <-- Label for the drop-down on the front-end
              "value": "8", <-- the id of the filter
            },
            {...}
          ];
}
*/

const filterPts = (mlsId, data) => {
  // Track used terms
  let hasR, hasL, hasF, hasLl, hasCs, hasCl, hasT, hasSf, hasC, hasM, hasCom;

  // Final returned array
  const filtersArr = [];

  const replaceSpaces = (str) => {
    return (str = str.split(" ").join("+"));
  };

  // Format the value value, push into the object
  const formatptID = (term, ptID, type) => {
    const el = {
      display: term,
      value: null,
    };

    if (type === "pt") {
      el.value = ptID;
    } else if (type === "st") {
      const stString = replaceSpaces(term);
      el.value = `${ptID}&a_propSubType%5B%5D=${stString}`;
    }

    filtersArr.push(el);
  };

  // If it's a match, send to formatting
  const checkMatch = (term, ptID, type) => {
    // Residential
    if (term === "Residential" && hasR !== true) {
      formatptID(term, ptID, type);
      hasR = true;
    }
    // Residential Lease
    if (term === "Residential Lease" && hasL !== true) {
      formatptID(term, ptID, type);
      hasL = true;
    }
    // Townhouse
    else if (term === "Townhouse" && hasT !== true) {
      formatptID(term, ptID, type);
      hasT = true;
    }
    // Single Familty Residence
    else if (term.includes("Single Family Residence") && hasSf !== true) {
      formatptID(term, ptID, type);
      hasSf = true;
    }
    // Condo
    else if (term.includes("Condo") && hasC !== true) {
      formatptID(term, ptID, type);
      hasC = true;
    }
    // Manufactured
    else if (term.includes("Manufactured On Land") && hasM !== true) {
      formatptID(term, ptID, type);
      hasM = true;
    }
    // Lots & Land
    else if (
      (term.includes("Land") || term.includes("Lots")) &&
      hasLl !== true
    ) {
      formatptID(term, ptID, type);
      hasLl = true;
    }
    // Farms
    else if (term.includes("Farms") && hasF !== true) {
      formatptID(term, ptID, type);
      hasF = true;
    }
    // Commercial Lease
    else if (
      term.includes("Commercial") &&
      term.includes("Sale") &&
      hasCs !== true &&
      hasCom !== true
    ) {
      formatptID(term, ptID, type);
      hasCs = true;
    }
    // Commercial Sale
    else if (
      term.includes("Commercial") &&
      term.includes("Lease") &&
      hasCl !== true &&
      hasCom !== true
    ) {
      formatptID(term, ptID, type);
      hasCl = true;
    }
    // Commercial - here, we'll take either cS + cL or just com
    else if (
      term === "Commercial" &&
      hasCom !== true &&
      hasCl !== true &&
      hasCs !== true
    ) {
      formatptID(term, ptID, type);
      hasCom = true;
    }
  };

  // Loop 1: loop through each object returned from the API call
  for (let i = 0; i < data.length; i++) {
    const obj = data[i]; // obj
    const ptId = obj.mlsPtID; // mlsPtID
    const pT = obj.mlsPropertyType; // property type term
    // check if it's a match
    checkMatch(pT, ptId, "pt");

    const subArr = JSON.parse(obj.propSubTypes);

    // loop through the propSubTypes
    for (let st = 0; st < subArr.length; st++) {
      // check if it's a match
      const sT = subArr[st]; // property type term
      checkMatch(sT, ptId, "st");
    }
  }

  return { mlsId: mlsId, values: filtersArr };
};

module.exports = { filterPts };
