/*
Data from the IDX Broker API has duplicates of what we want on the front end. The duplicates come from 
different "Property Type" id's included in the repsonse, but we don't need this for the omnibar. 
These helper functions filter out the extra objects and returns an array of formatted objs like: 
[
  { 
   lable: "Lebanon, TN", <-- the label is what we use to display the option in the drop-down on the front-end
   value: "25926", <-- this is the city's id
   type: "city", <-- this specifies the type (city, zip, county)
   mlsId: "b022", 
  },
  {...}
];
*/

const filterCounties = (mlsId, data) => {
  // Store Strings of each formatted county
  const formattedCounties = [];

  data.forEach((county) =>
    formattedCounties.push(
      JSON.stringify({
        lable: county.countyName,
        value: county.countyID,
        type: "county",
        mlsId: mlsId,
      })
    )
  );

  // Filter out duplicate strings (stringified county objs)
  const uniqueCounties = [...new Set(formattedCounties)];

  // Holds the final arr of objs (loop through and parse)
  const finalArr = [];
  uniqueCounties.forEach((county) => {
    if (county !== null) {
      finalArr.push(JSON.parse(county));
    }
  });

  return finalArr;
};

const filterZips = (mlsId, data) => {
  // Store Strings of each formatted county
  const formattedZips = [];

  data.forEach((zip) =>
    formattedZips.push(
      JSON.stringify({
        lable: zip.zipcode,
        value: zip.zipcode,
        type: "zipcode",
        mlsId: mlsId,
      })
    )
  );

  // Filter out duplicate strings (stringified zip objs)
  const uniqueZips = [...new Set(formattedZips)];

  // Holds the final arr of objs (loop through and parse)
  const finalArr = [];
  uniqueZips.forEach((zip) => {
    if (zip !== null) {
      finalArr.push(JSON.parse(zip));
    }
  });

  return finalArr;
};

const filterCities = (mlsId, data) => {
  // Store Strings of each formatted city
  const formattedCities = [];

  data.forEach((city) =>
    formattedCities.push(
      JSON.stringify({
        lable: `${city.cityName}, ${city.stateAbrv}`,
        value: city.cityID,
        type: "city",
        mlsId: mlsId,
      })
    )
  );

  // Filter out duplicate strings (stringified city objs)
  const uniqueCities = [...new Set(formattedCities)];

  // Holds the final arr of objs (loop through and parse)
  const finalArr = [];
  uniqueCities.forEach((city) => {
    if (city !== null) {
      finalArr.push(JSON.parse(city));
    }
  });

  return finalArr;
};

module.exports = { filterCounties, filterZips, filterCities };
